/*
 * Smart GWT (GWT for SmartClient)
 * Copyright 2008 and beyond, Isomorphic Software, Inc.
 *
 * Smart GWT is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3
 * as published by the Free Software Foundation.  Smart GWT is also
 * available under typical commercial license terms - see
 * http://smartclient.com/license
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 */
/* sgwtgen */
 
package com.smartgwt.client.util;


import com.smartgwt.client.event.*;
import com.smartgwt.client.core.*;
import com.smartgwt.client.types.*;
import com.smartgwt.client.data.*;
import com.smartgwt.client.data.Record;
import com.smartgwt.client.data.events.*;
import com.smartgwt.client.browser.window.*;
import com.smartgwt.client.rpc.*;
import com.smartgwt.client.ai.*;
import com.smartgwt.client.callbacks.*;
import com.smartgwt.client.tools.*;
import com.smartgwt.client.bean.*;
import com.smartgwt.client.widgets.*;
import com.smartgwt.client.widgets.ai.*;
import com.smartgwt.client.widgets.events.*;
import com.smartgwt.client.widgets.form.*;
import com.smartgwt.client.widgets.form.validator.*;
import com.smartgwt.client.widgets.form.fields.*;
import com.smartgwt.client.widgets.tile.*;
import com.smartgwt.client.widgets.tile.events.*;
import com.smartgwt.client.widgets.grid.*;
import com.smartgwt.client.widgets.grid.events.*;
import com.smartgwt.client.widgets.chart.*;
import com.smartgwt.client.widgets.layout.*;
import com.smartgwt.client.widgets.layout.events.*;
import com.smartgwt.client.widgets.menu.*;
import com.smartgwt.client.widgets.tour.*;
import com.smartgwt.client.widgets.notify.*;
import com.smartgwt.client.widgets.rte.*;
import com.smartgwt.client.widgets.rte.events.*;
import com.smartgwt.client.widgets.ace.*;
import com.smartgwt.client.widgets.ace.events.*;
import com.smartgwt.client.widgets.tab.*;
import com.smartgwt.client.widgets.toolbar.*;
import com.smartgwt.client.widgets.tree.*;
import com.smartgwt.client.widgets.tree.events.*;
import com.smartgwt.client.widgets.tableview.*;
import com.smartgwt.client.widgets.viewer.*;
import com.smartgwt.client.widgets.calendar.*;
import com.smartgwt.client.widgets.calendar.events.*;
import com.smartgwt.client.widgets.cube.*;
import com.smartgwt.client.widgets.notify.*;
import com.smartgwt.client.widgets.drawing.*;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.gwt.event.shared.*;
import com.google.gwt.dom.client.Element;
import com.google.gwt.user.client.ui.Widget;
import com.google.gwt.core.client.JavaScriptObject;

import com.smartgwt.client.util.*;
import com.smartgwt.client.util.events.*;
import com.smartgwt.client.util.workflow.*;
import com.smartgwt.client.util.workflow.Process; // required to override java.lang.Process
import com.smartgwt.client.util.tour.*;


/**
 * Utilities for working with JSON data. <P> <h3>Native JSON.stringify() vs isc.JSON.encode()</h3> <P> For maximum
 * performance when serializing large datasets, consider using the native <code>JSON.stringify()</code> when your data
 * meets certain criteria. Native serialization is approximately <b>6-10x faster</b> than <code>isc.JSON.encode()</code>,
 * but lacks several features that Smart GWT provides: <P> <b>Use native <code>JSON.stringify()</code> when ALL of these
 * conditions are met:</b> <ul> <li>Data contains <b>no circular references</b> - native throws an error on cycles rather  
 * than handling them gracefully. If your data might have cycles (e.g., parent/child     back-references in tree
 * structures), you must use <code>isc.JSON.encode()</code>.</li> <li>Data contains <b>no Date values</b>, OR you accept
 * JSON's default date handling.     Native <code>JSON.stringify()</code> converts Dates to ISO 8601 strings     (e.g.,
 * "2024-01-15T14:30:00.000Z"). This loses:     <ul>     <li><b>Logical Date distinction</b>: Smart GWT's logical dates
 * (date-only, no time         component) are serialized identically to datetimes by native JSON</li>     <li><b>Logical
 * Time distinction</b>: Smart GWT's logical times (time-only, no date         component) become full datetime strings</li>
 * <li><b>Timezone handling</b>: Native always outputs UTC; Smart GWT can preserve         local timezone context for
 * logical dates/times</li>     </ul>     Use {@link com.smartgwt.client.util.JSONEncoder#getDateFormat
 * JSONEncoder.dateFormat} with values like "logicalDateConstructor" to preserve     these distinctions when round-tripping
 * data.</li> <li>Data contains <b>no Smart GWT class instances</b> - native JSON cannot serialize     Smart GWT widgets or
 * other framework objects meaningfully</li> <li>You do <b>not need pretty-printed output</b> - native JSON.stringify's
 * optional     formatting is less flexible than {@link com.smartgwt.client.util.JSONEncoder#getPrettyPrint
 * JSONEncoder.prettyPrint}</li> <li>You do <b>not need custom serialization</b> via object <code>_serialize()</code>
 * methods</li> </ul> <P> <b>Use <code>isc.JSON.encode()</code> when ANY of these apply:</b> <ul> <li>Data might contain
 * circular references (automatically detected and handled)</li> <li>You need to preserve Smart GWT's logical
 * date/time/datetime distinctions</li> <li>You're serializing Smart GWT components or class instances</li> <li>You need
 * back-reference paths for reconstructing object graphs</li> <li>You want configurable date formats ({@link
 * com.smartgwt.client.types.JSONDateFormat})</li> <li>You need readable, pretty-printed output</li> </ul>
 */
@BeanFactory.FrameworkClass
public class JSON {



    // ********************* Properties / Attributes ***********************

    // ********************* Methods ***********************

    // ********************* Static Methods ***********************


	/**
     * Decodes strict JSON using native browser JSON parsing APIs. This API will not work with pseudo-JSON that must be parsed
     * as JavaScript (such as that produced by encoding with {@link com.smartgwt.client.util.JSONEncoder#getDateFormat
     * JSONEncoder.dateFormat} "dateConstructor" or "logicalDateConstructor"). <P> This API is called "safe" because using a
     * JavaScript eval() to decode JSON is potentially  unsafe if the data being decoded is untrusted. For example, if users
     * are able to save data  as JSON for other uses to see (such as sharing {@link
     * com.smartgwt.client.widgets.grid.ListGrid#getViewState ListGrid.viewState}) and there is no  validation of the saved
     * data to ensure safety and some users are untrusted, then saved JSON  could be used similarly to an XSS attack, allowing
     * one user to execute JavaScript code in  another user's browser. <P> Note that, because JSON has no way of representing
     * dates, serializing a structure that contains  dates and then deserializing with decodeSafe() necessarily results in any
     * Dates becoming strings.   Use {@link com.smartgwt.client.util.JSONEncoder#getDateFormat JSONEncoder.dateFormat}
     * "logicalDateString" in combination with {@link com.smartgwt.client.util.JSON#decodeSafeWithDates decodeSafeWithDates()}
     * to round-trip date, time and datetime values accurately.
     * @param jsonString JSON data to be de-serialized
     * @param jsonReviver optional setting
     *
     * @return object derived from JSON String
     */
    public static native Map decodeSafe(String jsonString, Function jsonReviver) /*-{
        var ret = $wnd.isc.JSON.decodeSafe(jsonString, 
			$entry( function() { 
				if(jsonReviver!=null) jsonReviver.@com.smartgwt.client.core.Function::execute()(
				);
			}));
        return @com.smartgwt.client.util.JSOHelper::convertToMap(Lcom/google/gwt/core/client/JavaScriptObject;)(ret);
    }-*/;


	/**
     * Uses {@link com.smartgwt.client.util.JSON#decodeSafe decodeSafe()} to decode strict JSON using native browser JSON
     * parsing APIs,  with settings to correctly process formatted date values created with {@link
     * com.smartgwt.client.util.JSONEncoder#getDateFormat JSONEncoder.dateFormat} "logicalDateString".
     * @param jsonString JSON data to be de-serialized
     */
    public static native void decodeSafeWithDates(String jsonString) /*-{
        $wnd.isc.JSON.decodeSafeWithDates(jsonString);
    }-*/;


	/**
     * De-serialize an object from JSON, attempting native JSON.parse() first for performance, then falling back to eval() if
     * the input contains Smart GWT pseudo-JSON extensions (unquoted keys, single quotes, trailing commas, etc.). <P> This
     * method is optimized for cases where the JSON is typically strict (conforming to the JSON spec) but may occasionally
     * contain pseudo-JSON extensions. The native JSON.parse() fast path provides significant performance improvements in V8
     * and other modern engines. <P> For data that is always strict JSON, use {@link com.smartgwt.client.util.JSON#decodeSafe
     * decodeSafe()} instead.
     * @param jsonString JSON or pseudo-JSON data to be de-serialized
     *
     * @return object derived from JSON String
     */
    public static native Map decodeStrict(String jsonString) /*-{
        var ret = $wnd.isc.JSON.decodeStrict(jsonString);
        return @com.smartgwt.client.util.JSOHelper::convertToMap(Lcom/google/gwt/core/client/JavaScriptObject;)(ret);
    }-*/;



    // ***********************************************************


    /**
     * Serialize an object as a JSON string. <P> Automatically handles circular references - see {@link
     * com.smartgwt.client.util.JSONEncoder#getCircularReferenceMode circularReferenceMode}.
     * <P>
     * Note that using the String produced by this API with {@link com.smartgwt.client.util.JSON#decode JSON.decode} <b>will not
     * successfully preserve dates</b>. Use {@link com.smartgwt.client.util.JSONEncoder#setDateFormat JSONEncoder.setDateFormat} "dateConstructor" to have
     * dates round-trip properly.
     * <P>
     * Because GWT does not support Java reflection, JSON encoding cannot
     * discover the properties of an arbitrary Java POJO.  The following 
     * objects are supported:
     * <ul>
     * <li> any primitive type (String, Date, Number, Boolean)
     * <li> any Map or Collection in any level of nesting
     * <li> DataClass (Record's superclass) and RecordList
     * <li> any widget (see +link{JSONEncoder.serializeInstances})
     * <li> JavaScriptObject
     * <li> an Array containing any of the above
     * </ul>
     *
     * @param object object to serialize
     * @return object encoded as a JSON String
     */
    public static native String encode(JavaScriptObject object) /*-{
        return $wnd.isc.JSON.encode(object);
    }-*/;

    /**
     * Serialize an object as a JSON string. <P> Automatically handles circular references - see {@link
     * com.smartgwt.client.util.JSONEncoder#getCircularReferenceMode circularReferenceMode}.
     * <P>
     * Note that using the String produced by this API with {@link com.smartgwt.client.util.JSON#decode JSON.decode} <b>will not
     * successfully preserve dates</b>. Use {@link com.smartgwt.client.util.JSONEncoder#setDateFormat JSONEncoder.setDateFormat} "dateConstructor" to have
     * dates round-trip properly.
     * <P>
     * Because GWT does not support Java reflection, JSON encoding cannot
     * discover the properties of an arbitrary Java POJO.  The following 
     * objects are supported:
     * <ul>
     * <li> any primitive type (String, Date, Number, Boolean)
     * <li> any Map or Collection in any level of nesting
     * <li> DataClass (Record's superclass) and RecordList
     * <li> any widget (see +link{JSONEncoder.serializeInstances})
     * <li> JavaScriptObject
     * <li> an Array containing any of the above
     * </ul>
     * 
     * @param object object to serialize
     * @param settings optional settings for encoding
     *
     * @return object encoded as a JSON String
     */
    public static String encode(JavaScriptObject object, JSONEncoder settings) {
        if (settings == null) return encode(object);
        return settings.encode(object);
    }

    /**
     * De-serialize an object from JSON.  Currently, this is simply a JavaScript eval() and should be used for trusted data
     * only.
     * @param jsonString JSON data to be de-serialized
     *
     * @return object derived from JSON String
     */
    public static native JavaScriptObject decode(String jsonString) /*-{
        return $wnd.isc.JSON.decode(jsonString);
    }-*/;

}
